/*---------------------------------------------------------------------------*
  Project:  EVL0 system file generator
  File:     makedlf.c

  Copyright 2009 Nintendo.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.

  $Log: makedlf.c $
  $NoKeywords: $
 *---------------------------------------------------------------------------*/

//#include <getopt.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <types.h>

#define DLFFILE         "boot.dlf"
#define HEADERSIZE      32
#define MAXSIZE         (1024*1024)

#define REV32(x)            ( (((x) & 0x000000ff) << 24)          \
                          | ( ((x) & 0x0000ff00) << 8)       \
                          | ( ((x) & 0x00ff0000) >> 8)       \
                          | ( ((x) & 0xff000000) >> 24) )

#define RoundUp32(n) (((u32)(n) + 31) & ~((u32)31))

#define MAX_DLF_ENTRY  256

char* AppName;
static BOOL Quiet = FALSE;

static void usage(int err)
{
    printf("usage: %s [args]\n", AppName);
    printf("  -s esf                     # esf file name\n");
    printf("  -p <process1,process2,...> # process file list\n");
    printf("  -l <offset1,offset2,...>   # offset list\n");
    printf("  -o output                  # output file\n");
    printf("  -c                         # compat\n");
    printf("  -q                         # quiet\n");
    exit(err);
}

/*---------------------------------------------------------------------------*
  Name:         MakeDlf1

  Description:  TBD

  Arguments:    TBD

  Returns:      pointer to RAM image
 *---------------------------------------------------------------------------*/
static void MakeDlf1(u32 num, char** processlist, u32* lengthlist,
    u32 numOffset, u32* offsetList, char* outname)
{
    FILE*        fp;
    u32          i;
    u32          diskOffset = 0;

    // Blast out the entire image to the file
    fp = fopen(outname, "w");
    if (fp == NULL)
    {
        fprintf(stderr, "%s: unable to open img output file\n", AppName);
        exit(EXIT_FAILURE);
    }

    fprintf(fp, "v1.00\n");

    for(i = 0; i < num; i++)
    {
        if (0 < numOffset)
        {
            fprintf(fp, "0x00000000%08X,\"%s\"\n", offsetList[i], processlist[i]);
            diskOffset = RoundUp32(offsetList[i] + lengthlist[i]);
            --numOffset;
        }
        else
        {
            fprintf(fp, "0x00000000%08X,\"%s\"\n", diskOffset, processlist[i]);
            diskOffset = RoundUp32(diskOffset + lengthlist[i]);
        }
    }

    fprintf(fp, "0x00000000%08X,\"\"\n", diskOffset);

    fclose(fp);

    return;
}

/*---------------------------------------------------------------------------*
  Name:         MakeDlf2

  Description:  TBD

  Arguments:    TBD

  Returns:      pointer to RAM image
 *---------------------------------------------------------------------------*/
static void MakeDlf2(char* esfname, u32 num, char** processlist, u32* lengthlist, char* outname)
{
    FILE*        fp;
    u32          i;
    u32          offset = 0;

    // Blast out the entire image to the file
    fp = fopen(outname, "w");
    if (fp == NULL)
    {
        printf("%s: unable to open img output file\n", AppName);
        exit(EXIT_FAILURE);
    }

    fprintf(fp, "v2.00\n");

    fprintf(fp, "RVL_CONFIG_FILE,\"%s\"\n", esfname);

    for(i = 0; i < num; i++)
    {
        fprintf(fp, "0x00000000%08X,\"%s\"\n", offset, processlist[i]);
        offset = RoundUp32(offset + lengthlist[i]);
    }

    fprintf(fp, "0x00000000%08X,\"\"\n", offset);

    fclose(fp);

    return;
}

void ConvertToUnixFmt(char* fmt)
{
    if (fmt == NULL)
        return;

    if ( (*fmt == '/') && (*(fmt + 1) == '/') )
    {
        *fmt       = *(fmt + 2);
        *(fmt + 1) = ':';

        fmt += 3;
        while (*fmt != '\0')
        {
            *(fmt - 1) = *fmt;
            fmt++;
        }

        *(fmt - 1) = '\0';
    }
}


int main(int argc, char **argv)
{
    u32       num = 0;
    u32       numOffset = 0;
    char*     processList[MAX_DLF_ENTRY];
    u32       lengthList [MAX_DLF_ENTRY];
    u32       offsetList[MAX_DLF_ENTRY];

    char*     esfName = NULL;

    char*     outName = DLFFILE;

    u32       size = 0;

    char*     token = NULL;
	char*	  optarg = NULL;
	char*	  cur = NULL;
    FILE*     fp = NULL;
    s32       errFlg = 0;
    s32       c;
	s32		  argIdx = 1;
    void*     image;
    FILE*     file;

    s32       compatFlg = 0;
	int i = 0;

    memset(processList, 0, sizeof(u32)*256);
    memset(lengthList,  0, sizeof(u32)*256);

    AppName = argv[0];
	while(argIdx < argc)
	{		
		
		if(argc == 1)				
		{
			c = '?';
		}
		else
		{
			cur = argv[argIdx++];
		
			if((strlen(cur) == 2) && (cur[0] == '-'))
			{
				c = cur[1];
				if(argIdx < argc && ((argv[argIdx])[0] != '-') )
				{
					optarg = argv[argIdx++];
					if (optarg[0] == '"')
					{
						int tempArgIdx = argIdx;

					}					

				}
			}		
			else
			{
				c = '?';
			}
		}

		
        switch (c)
        {
          case 'q':
            Quiet = TRUE;
            break;

          case 'o':
            outName = optarg;
            break;

          case 's':
            esfName = optarg;
            if (!Quiet)
                printf("esf file name: %s\n", esfName);
            break;

          case 'p':
            token = strtok(optarg, ",");
            while (token)
            {
                ConvertToUnixFmt(token);
                if (!Quiet)
                    printf("process %d:     %s ", num+1, token);
                fp = fopen(token, "rb");
                if ( fp == NULL )
                {
                    fprintf(stderr, "unable to open %s\n", token);
                    errFlg++;
                    break;
                }
                processList[num] = token;
                fseek(fp, 0, SEEK_END);
                lengthList[num] = ftell(fp);
                fclose(fp);
                if (!Quiet)
                    printf("(0x%08x)\n", lengthList[num]);
                num++;
                token = strtok(NULL, ",");
            }
            break;

          case 'h':
            usage(0);
            break;

          case 'c':
            compatFlg = 1;
            break;

          case 'l':
            token = strtok(optarg, ",");
            while (token)
            {
                offsetList[numOffset] = strtoul(token, NULL, 0);
                if (!Quiet)
                    printf("offset %d:     0x%08x\n", numOffset+1, offsetList[numOffset]);
                numOffset++;
                token = strtok(NULL, ",");
            }
            break;

          default:
            fprintf(stderr, "unknown option %c\n", c);
            errFlg++;
            break;
        }
	}

    if (!Quiet)
    {
        printf("EVL0 disk layout file generator. Copyright 2009-2010 Nintendo.\n");
        printf("Built : %s %s\n", __DATE__, __TIME__);
    }

    if (compatFlg)
    {
        if (!esfName || !num || !outName || errFlg)
        {
            usage(EXIT_FAILURE);
        }

        ConvertToUnixFmt(outName);

        MakeDlf2(esfName, num, processList, lengthList, outName);
    }
    else
    {
        if (!num || !outName || errFlg)
        {
            usage(EXIT_FAILURE);
        }

        ConvertToUnixFmt(outName);

        MakeDlf1(num, processList, lengthList, numOffset, offsetList, outName);
    }

    if (!Quiet)
        printf("Generated %s\n", outName);

    return 0;
}
